<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Monthly Revenue By Type Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Jason Yates <jason@clientexec.com>
 * @license  ClientExec License
 * @version  1.2
 * @link     http://www.clientexec.com
 *
 *************************************************
 *   1.0 Initial Report Released.  - Alberto Vasquez
 *   1.2 Updated report to include a title & PEAR commenting
 ************************************************
 */

require_once 'modules/billing/models/Currency.php';
require_once 'modules/billing/models/BillingType.php';

/**
 * Monthly_Revenue_By_Type Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Jason Yates <jason@clientexec.com>
 * @license  ClientExec License
 * @version  1.2
 * @link     http://www.clientexec.com
 */
class Monthly_Revenue_By_Type extends Report
{
    private $lang;

    protected $featureSet = 'billing';
    public $hasgraph = true;

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Monthly Revenue By Type');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        //Billing Cycles
        $billingcycles = array();

        include_once 'modules/billing/models/BillingCycleGateway.php';
        $gateway = new BillingCycleGateway();
        $iterator = $gateway->getBillingCycles(array(), array('order_value', 'ASC'));

        while ($cycle = $iterator->fetch()) {
            $billingcycles[$cycle->id] = array(
                'name'            => $this->user->lang($cycle->name),
                'time_unit'       => $cycle->time_unit,
                'amount_of_units' => $cycle->amount_of_units
            );
        }
        //Billing Cycles

        include_once 'modules/admin/models/StatusAliasGateway.php' ;

        // Set the report information
        $this->SetDescription($this->user->lang('Displays total recurring transactions broken down by package types with the sum and expected monthly revenue from each.'));

        // Load the currency information
        $currency = new Currency($this->user);

        $currencyCode = ((isset($_REQUEST['currencycode']))? $_REQUEST['currencycode'] : $this->settings->get('Default Currency'));
        $currencyName = $currency->getName($currencyCode);

        $filter = '<form id="report" method="GET">'
            .'    <div style="text-align:center">'
            .'        '.$this->user->lang('Currency').': '
            .'        <select name="currencycode" id="currencycode" value="'.CE_Lib::viewEscape($currencyCode).'" onChange="ChangeTable(this.value);"> ';

        //Get all currencies of all users
        $result = Capsule::table('users as u')
            ->join('currency as c', 'c.abrv', '=', 'u.currency')
            ->select('c.abrv', 'c.name')
            ->distinct()
            ->orderBy('c.name', 'asc')
            ->get();

        $isSelectedCurrencyInTheList = false;

        foreach ($result as $row) {
            if (!$isSelectedCurrencyInTheList && $currencyName < $row->name) {
                $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
                $isSelectedCurrencyInTheList = true;
            } elseif ($currencyCode == $row->abrv) {
                $isSelectedCurrencyInTheList = true;
            }
            $filter .= '<option value="'.$row->abrv.'" '.(($currencyCode == $row->abrv)? 'selected' : '').'>'.$row->name.'</option>';
        }
        if (!$isSelectedCurrencyInTheList) {
            $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
            $isSelectedCurrencyInTheList = true;
        }

        $filter .= '</select>';
        $filter .= '    </div>'
            .'</form>';

        $graphdata = @$_GET['graphdata'];

        $userStatuses = StatusAliasGateway::userActiveAliases($this->user);
        $packageStatuses = StatusAliasGateway::packageActiveAliases($this->user);

        //SQL to generate the the result set of the report
        $result = Capsule::table('domains as d')
            ->leftJoin('recurringfee as rf', function ($join) {
                $join->on('rf.appliestoid', '=', 'd.id')
                     ->where('rf.billingtypeid', -1);
            })
            ->leftJoin('object_customField as ocf', function ($join) {
                $join->on('ocf.objectid', '=', 'd.id')
                     ->where('ocf.customFieldId', function ($query) {
                         $query->select('cf.id')
                               ->from('customField as cf')
                               ->where('cf.groupId', 2)
                               ->where('cf.subGroupId', 3)
                               ->where('cf.name', 'Domain Name')
                               ->limit(1);
                     });
            })
            ->join('users as u', 'd.customerid', '=', 'u.id')
            ->selectRaw('
                COUNT(*) AS counted,
                d.Plan,
                rf.paymentterm,
                d.use_custom_price,
                d.custom_price,
                ocf.value AS domain_name,
                u.currency
            ')
            ->where('rf.paymentterm', '<>', 0)
            ->whereRaw('IFNULL(rf.recurring, 0) <> 0')
            ->whereIn('u.status', $userStatuses)
            ->whereIn('d.status', $packageStatuses)
            ->groupBy('u.currency', 'rf.paymentterm', 'd.Plan', 'd.use_custom_price', 'd.custom_price')
            ->orderBy('u.currency')
            ->get();

        $expectedrevenueTotal = array();
        $sumpertotalpackagesTotal = array();

        include_once 'modules/billing/models/Prices.php';
        $prices = new Prices();

        // Just in case we have domains
        $dng = new DomainNameGateway($this->user);

        $aGroup = array();

        foreach ($result as $row) {
            $tCurrency = $row->currency;

            $result2 = Capsule::table('package as p')
                ->join('promotion as t', 't.id', '=', 'p.planid')
                ->select('t.name', 'p.planname', 'p.pricing as price', 't.type')
                ->where('p.id', $row->Plan)
                ->get();

            foreach ($result2 as $row2) {
                //Now we do the math, and add the class array and variables
                $row2->price = $prices->getPricing(PRODUCT_PRICE, $row->Plan, $tCurrency, $row2->price);
                $pricing = unserialize($row2->price);

                $packagePrice = 0;

                if (is_array($pricing)) {
                    if ($row2->type == 3) {  // Domain Type
                        $aDomainName = $dng->splitDomain($row->domain_name);
                        $tld = strtolower($aDomainName[1]);

                        $pricingInformation = array();

                        foreach ($pricing as $key => $value) {
                            $pricingInformation[$key] = $value;
                        }

                        $pricingArray = array_pop($pricingInformation['pricedata']);

                        if ($row->paymentterm != 0 && $billingcycles[$row->paymentterm]['time_unit'] == 'y' && $billingcycles[$row->paymentterm]['amount_of_units'] != 0 && isset($pricingArray[$row->paymentterm]['price'])) {
                            $packagePrice = ($pricingArray[$row->paymentterm]['price']/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                        }
                    } else {
                        if ($row->paymentterm != 0 &&  $billingcycles[$row->paymentterm]['amount_of_units'] != 0 && isset($pricing['price'.$row->paymentterm]) && is_numeric($pricing['price'.$row->paymentterm])) {
                            switch ($billingcycles[$row->paymentterm]['time_unit']) {
                                case 'd':
                                    $packagePrice = ($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])*30;
                                    break;
                                case 'w':
                                    $packagePrice = (($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])/7)*30;
                                    break;
                                case 'm':
                                    $packagePrice = $pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'];
                                    break;
                                case 'y':
                                    $packagePrice = ($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                                    break;
                            }
                        }
                    }
                }

                //this is for overrided prices
                if ($row->paymentterm != 0 && $row->use_custom_price && $billingcycles[$row->paymentterm]['amount_of_units'] != 0) {
                    switch ($billingcycles[$row->paymentterm]['time_unit']) {
                        case 'd':
                            $packagePrice = ($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])*30;
                            break;
                        case 'w':
                            $packagePrice = (($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])/7)*30;
                            break;
                        case 'm':
                            $packagePrice = $row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'];
                            break;
                        case 'y':
                            $packagePrice = ($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                            break;
                    }
                }

                $tMonthlyRevenue = $packagePrice * $row->counted;
                $tPackageType = $row2->name;
                $tPackageCount = $row->counted;

                if (isset($expectedrevenueTotal[$tCurrency])) {
                    $expectedrevenueTotal[$tCurrency] += $tMonthlyRevenue;
                } else {
                    $expectedrevenueTotal[$tCurrency] = $tMonthlyRevenue;
                }

                if (isset($sumpertotalpackagesTotal[$tCurrency])) {
                    $sumpertotalpackagesTotal[$tCurrency] += $tPackageCount;
                } else {
                    $sumpertotalpackagesTotal[$tCurrency] = $tPackageCount;
                }

                if (isset($aGroup[$tCurrency][$tPackageType])) {
                    $tArray = $aGroup[$tCurrency][$tPackageType];
                    $aGroup[$tCurrency][$tPackageType] = array($tPackageType,$tArray[1]+$tPackageCount,$tArray[2]+$tMonthlyRevenue);
                } else {
                    $aGroup[$tCurrency][$tPackageType] = array($tPackageType,$tPackageCount,$tMonthlyRevenue);
                }

                // NOTE: remember the addon cycle can be different than the package's
                $result3 = Capsule::table('recurringfee as rf')
                    ->leftJoin('users as u', 'u.id', '=', 'rf.customerid')
                    ->leftJoin('domains as d', function ($join) {
                        $join->on('rf.appliestoid', '=', 'd.id')
                             ->leftJoin('recurringfee as rrff', function ($subJoin) {
                                 $subJoin->on('rrff.appliestoid', '=', 'd.id')
                                         ->where('rrff.billingtypeid', -1);
                             });
                    })
                    ->selectRaw('COUNT(*) AS counted, SUM(rf.amount * rf.quantity) AS total, rf.paymentterm')
                    ->where('rf.billingtypeid', BILLINGTYPE_PACKAGE_ADDON)
                    ->where('d.Plan', $row->Plan)
                    ->whereIn('d.status', $packageStatuses)
                    ->where('rrff.paymentterm', $row->paymentterm)
                    ->where('u.currency', $tCurrency)
                    ->groupBy('rf.paymentterm')
                    ->get();

                $groupName = "* $tPackageType add-ons";

                foreach ($result3 as $row3) {
                    $tAddonCount = $row3->counted;

                    //get expected monthly based on term
                    if ($row3->paymentterm != 0 && $billingcycles[$row3->paymentterm]['amount_of_units'] != 0) {
                        switch ($billingcycles[$row3->paymentterm]['time_unit']) {
                            case 'd':
                                $row3->total = ($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])*30;
                                break;
                            case 'w':
                                $row3->total = (($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])/7)*30;
                                break;
                            case 'm':
                                $row3->total = $row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'];
                                break;
                            case 'y':
                                $row3->total = ($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])/12;
                                break;
                        }
                    } else {
                        continue;
                    }
                    $tExpectedRevenue = $currency->format($tCurrency, $row3->total, true, false);

                    if (isset($expectedrevenueTotal[$tCurrency])) {
                        $expectedrevenueTotal[$tCurrency] += $row3->total;
                    } else {
                        $expectedrevenueTotal[$tCurrency] = $row3->total;
                    }

                    if (isset($aGroup[$tCurrency][$groupName])) {
                        $tArray = $aGroup[$tCurrency][$groupName];
                        $aGroup[$tCurrency][$groupName] = array($groupName, $tArray[1] + $tAddonCount, $tArray[2] + $row3->total);
                    } else {
                        $aGroup[$tCurrency][$groupName] = array($groupName, $tAddonCount, $row3->total);
                    }
                }
            }
        }

        if ($graphdata) {
            //this supports lazy loading and dynamic loading of graphs
            $this->reportData = $this->GraphData($aGroup[$currencyCode], $currencyCode);
            return;
        }

        $aGroupIds = array();

        if (isset($aGroup)) {
            foreach ($aGroup as $currencyKey => $aCurrency) {
                $grouphidden = true;
                $groupid = "id-" . $currencyKey;

                if ($currencyCode == $currencyKey) {
                    $grouphidden = false;
                }

                //Loop through group array and change price format now that all the sums have been made
                $aGroupWithCurrency = array();

                foreach ($aCurrency as $tGroup) {
                    $aGroupWithCurrency[] = array($tGroup[0], $tGroup[1], $currency->format($currencyKey, $tGroup[2], true));
                }

                $aNewGroup = $aGroupWithCurrency;

                $aNewGroup[] = array("","","");

                $expectedrevenueTotal[$currencyKey] = $currency->format($currencyKey, $expectedrevenueTotal[$currencyKey], true);
                $aNewGroup[] = array("<b>".$this->user->lang("Totals")."</b>","<b>".$sumpertotalpackagesTotal[$currencyKey]."</b>", "<b>".$expectedrevenueTotal[$currencyKey]."</b>");

                $this->reportData[] = array(
                    "group" => $aNewGroup,
                    "groupname" => $this->user->lang("Package Types").' ('.$currencyKey.')',
                    "label" => array($this->user->lang('Package Type'),$this->user->lang('Total Packages'),$this->user->lang('Expected Monthly Revenue')),
                    "groupId" => $groupid,
                    "isHidden" => $grouphidden
                );

                $aGroupIds[] = $groupid;

                unset($aNewGroup);
            }
        }

        $filter .= '<script type="text/javascript">'
            .'    function ChangeTable(currencycode){';

        foreach ($aGroupIds as $groupId) {
            $filter .= "if (document.getElementById('$groupId') != null) {\n"
                ."    document.getElementById('$groupId').style.display='none';\n"
                ."}\n";
        }

        $filter .= "        if (document.getElementById('id-'+currencycode) != null) {\n"
            ."            document.getElementById('id-'+currencycode).style.display='';\n"
            ."        };\n"
            ."        clientexec.populate_report('Monthly_Revenue_By_Type-Revenue','#myChart',{currencycode:currencycode});\n"
            .'    }'
            .'</script>';
        echo $filter;
    }


    /**
     * Function to output the xml for the graph data
     *
     * @return XML - graph data
     */
    function GraphData($aGroups, $currencyCode)
    {
        //get default currency symbol
        $currency = new Currency($this->user);

        $graph_data = array(
              "xScale" => "ordinal",
              "yScale" => "exponential",
              "xType" => "number",
              "yType" => "currency",
              "yPre" => $currency->ShowCurrencySymbol($currencyCode),
              "yFormat" => "addcomma",
              "type" => "bar",
              "main" => array());

        $group_data = array();
        $group_data['className'] = ".report";
        $group_data['data'] = array();

        $index = 0;

        foreach ($aGroups as $group) {
            $pretty_total = $currency->format($currencyCode, $group[2], true);

            $data = array();
            $data["x"] = array($index,substr($group[0], 0, 15));
            $data["y"] = $group[2];
            $data["tip"] = "<strong>".$group[0]."</strong><br/>".$group[1]." Packages for ".$pretty_total;
            $group_data["data"][] = $data;
            $index++;
        }
        $graph_data["main"][] = $group_data;
        return json_encode($graph_data);
    }

    /**
     * Function to return the average, used for making the pie chart
     *
     * @return num - average value
     */
    function ReturnAverage($count, $totalCount)
    {
        $avg = ($count/$totalCount);
        $avg =  $avg * 100;
        return ceil($avg);
    }
}
