<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Revenue Forecast Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 *
 *************************************************
 *   1.0 Initial Report Released.  - Juan D. Bolivar
 ************************************************
 */

require_once 'modules/admin/models/Package.php';
require_once 'modules/clients/models/UserPackage.php';
require_once 'modules/billing/models/Currency.php';
require_once 'modules/clients/models/User.php';
require_once 'modules/billing/models/Invoice.php';

/**
 * Revenue_Forecast Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 */
class Revenue_Forecast extends Report
{
    private $lang;

    protected $featureSet = 'billing';

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Revenue Forecast');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        //Billing Cycles
        $billingcycles = array();

        include_once 'modules/billing/models/BillingCycleGateway.php';
        $gateway = new BillingCycleGateway();
        $iterator = $gateway->getBillingCycles(array(), array('order_value', 'ASC'));

        while ($cycle = $iterator->fetch()) {
            $billingcycles[$cycle->id] = array(
                'name'            => $this->user->lang($cycle->name),
                'time_unit'       => $cycle->time_unit,
                'amount_of_units' => $cycle->amount_of_units
            );
        }
        //Billing Cycles

        include_once 'modules/admin/models/StatusAliasGateway.php' ;

        // Set the report information
        $this->SetDescription($this->user->lang('Displays the total revenue forecast per month.'));

        @set_time_limit(0);

        // Load the currency information
        $currency = new Currency($this->user);

        $currencyCode = ((isset($_REQUEST['currencycode']))? $_REQUEST['currencycode'] : $this->settings->get('Default Currency'));
        $currencyName = $currency->getName($currencyCode);

        $filter = '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'
            .'        '.$this->user->lang('Currency').': '
            .'        <select class="normalSelect2 w-25" name="currencycode" id="currencycode" value="'.CE_Lib::viewEscape($currencyCode).'" > ';

        $isSelectedCurrencyInTheList = false;

        //Get all currencies of all users
        $result = Capsule::table('users as u')
            ->join('currency as c', 'c.abrv', '=', 'u.currency')
            ->select('c.abrv', 'c.name')
            ->distinct()
            ->orderBy('c.name', 'ASC')
            ->get();

        foreach ($result as $row) {
            if (!$isSelectedCurrencyInTheList && $currencyName < $row->name) {
                $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
                $isSelectedCurrencyInTheList = true;
            } elseif ($currencyCode == $row->abrv) {
                $isSelectedCurrencyInTheList = true;
            }

            $filter .= '<option value="'.$row->abrv.'" '.(($currencyCode == $row->abrv)? 'selected' : '').'>'.$row->name.'</option>';
        }

        if (!$isSelectedCurrencyInTheList) {
            $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
            $isSelectedCurrencyInTheList = true;
        }

        $filter .= '</select>';

        $getMonthDetails = @$_GET['getMonthDetails'];

        if ($getMonthDetails) {
            $this->getMonthDetails($currencyCode);

            exit;
        }

        $userStatuses = StatusAliasGateway::userActiveAliases($this->user);
        $packageStatuses = StatusAliasGateway::getInstance($this->user)->getPackageStatusIdsFor(array(PACKAGE_STATUS_PENDING, PACKAGE_STATUS_ACTIVE));
        $amountOfMonths = ((isset($_REQUEST['amountOfMonths']))? $_REQUEST['amountOfMonths'] : 12);
        $masterGroup = array();

        //Get recurring fees
        $result = Capsule::table('recurringfee as r')
            ->join('users as u', 'r.customerid', '=', 'u.id')
            ->leftJoin('domains as d', 'r.appliestoid', '=', 'd.id')
            ->select(
                'r.customerid',
                'r.amount',
                'r.quantity',
                'r.nextbilldate',
                'r.appliestoid',
                'r.billingtypeid',
                'r.taxable',
                'r.paymentterm'
            )
            ->whereNotNull('r.nextbilldate')
            ->whereRaw("DATE_FORMAT(r.nextbilldate, '%Y%m') <= DATE_FORMAT(DATE_ADD(CURDATE(), INTERVAL ? MONTH), '%Y%m')", [$amountOfMonths - 1])
            ->whereIn('u.status', $userStatuses)
            ->where(function ($query) use ($packageStatuses) {
                $query->where('r.appliestoid', '=', 0)
                    ->orWhere(function ($sub) use ($packageStatuses) {
                        $sub->where('r.appliestoid', '!=', 0)
                            ->whereIn('d.status', $packageStatuses)
                            ->where(function ($inner) {
                                $inner->where('r.billingtypeid', '!=', BILLINGTYPE_PACKAGE)
                                    ->orWhere(function ($deep) {
                                        $deep->where('r.billingtypeid', '=', BILLINGTYPE_PACKAGE)
                                            ->whereRaw('IFNULL(r.recurring, 0) = 1');
                                    });
                            });
                    });
            })
            ->where('r.paymentterm', '!=', 0)
            ->where('u.currency', '=', $currencyCode)
            ->orderBy('r.nextbilldate')
            ->get();

        foreach ($result as $row) {
            // Check for taxes
            $tax = 0;
            $customertax          = 0;
            $customertax2         = 0;
            $customertax2compound = 0;
            $user = new User($row->customerid);

            if ($user->IsTaxable() == 1) {
                //determine country and state and see if there is a tax in the rules to match
                $customertax  = $user->GetTaxRate();
                //-1 is returned when we don't have a taxrate
                $customertax = ($customertax == -1) ? 0 : (float) $customertax;

                $customertax2 = $user->GetTaxRate(2);
                //-1 is returned when we don't have a taxrate
                $customertax2 = ($customertax2 == -1) ? 0 : (float) $customertax2;

                if ($user->isTax2Compound()) {
                    $customertax2compound = 1;
                }
            }

            if ($row->appliestoid > 0 && $row->billingtypeid == -1) {
                $domain = new UserPackage($row->appliestoid);
                $paymentterm = $domain->getPaymentTerm();
                $taxable = $domain->isTaxable()? true : false;
                $amount = $domain->getPrice(false);
            } else {
                $paymentterm = $row->paymentterm;
                $taxable = $row->taxable;
                $amount = $row->amount;
            }

            // if $amount is not a numeric, skip this.
            if (!is_numeric($amount)) {
                continue;
            }

            if ($taxable) {
                $tax1 = round($amount * $row->quantity * $customertax / 100, 2);

                if ($customertax2compound) {
                    $tax2 = round(($amount * $row->quantity + $tax1) * $customertax2 / 100, 2);
                } else {
                    $tax2 = round($amount * $row->quantity * $customertax2 / 100, 2);
                }

                $tax = $currency->format($currencyCode, $tax1 + $tax2);
            }

            $masterGroup[] = array($row->nextbilldate, $amount * $row->quantity, $tax, $paymentterm);
        }

        $newMonthTotals  = array();

        //Initialize months revenue
        for ($monthsAhead = 0; $monthsAhead < $amountOfMonths; $monthsAhead++) {
            $newMonthTotals[date("Y M", mktime(0, 0, 0, date("m") + $monthsAhead, 1, date("Y")))] = array(
                'data'   => array(),
                'amount' => 0,
                'taxes'  => 0,
                'total'  => 0
            );
        }

        //Get amounts already invoiced
        $result = Capsule::table('invoice as i')
            ->select(
                'i.billdate',
                Capsule::raw('IFNULL(i.amount, 0) as amount'),
                Capsule::raw('IFNULL(i.subtotal, 0) as subtotal')
            )
            ->whereRaw("DATE_FORMAT(i.billdate, '%Y%m') >= DATE_FORMAT(CURDATE(), '%Y%m')")
            ->whereRaw("DATE_FORMAT(i.billdate, '%Y%m') <= DATE_FORMAT(DATE_ADD(CURDATE(), INTERVAL ? MONTH), '%Y%m')", [$amountOfMonths - 1])
            ->whereIn('i.status', [
                INVOICE_STATUS_UNPAID,
                INVOICE_STATUS_PAID,
                INVOICE_STATUS_PENDING,
                INVOICE_STATUS_PARTIALLY_PAID
            ])
            ->where('i.currency', $currencyCode)
            ->get();

        //Add invoices already generated
        foreach ($result as $row) {
            $monthAndYear = date("Y M", mktime(0, 0, 0, date("m", strtotime($row->billdate)), 1, date("Y", strtotime($row->billdate))));

            if (isset($newMonthTotals[$monthAndYear])) {
                $newMonthTotals[$monthAndYear]['amount'] += $row->subtotal;
                $newMonthTotals[$monthAndYear]['taxes']  += $row->amount - $row->subtotal;
                $newMonthTotals[$monthAndYear]['total']  += $row->amount;

                if (isset($newMonthTotals[$monthAndYear]['data']['Invoice'])) {
                    $newMonthTotals[$monthAndYear]['data']['Invoice'][1] += 1;
                    $newMonthTotals[$monthAndYear]['data']['Invoice'][2] += $row->subtotal;
                    $newMonthTotals[$monthAndYear]['data']['Invoice'][3] += $row->amount - $row->subtotal;
                    $newMonthTotals[$monthAndYear]['data']['Invoice'][4] += $row->amount;
                } else {
                    $newMonthTotals[$monthAndYear]['data']['Invoice'] = array(
                        'Invoice',
                        1,
                        $row->subtotal,
                        $row->amount - $row->subtotal,
                        $row->amount
                    );
                }
            }
        }

        $maxMonthAndYear = date("Ym", mktime(0, 0, 0, date("m") + $amountOfMonths - 1, 1, date("Y")));

        //Add recurring charges
        foreach ($masterGroup as $masterGroupinfo) {
            $nextbilldate    = $masterGroupinfo[0];
            $amount          = $masterGroupinfo[1];
            $taxes           = $masterGroupinfo[2];
            $paymentterm     = $masterGroupinfo[3];

            if ($paymentterm == 0) {
                break;
            }

            $daysAhead = 0;
            $monthsAhead = 0;
            $yearsAhead = 0;
            $currentMonthAndYear = 0;

            do {
                $monthAndYear = date("Y M", mktime(0, 0, 0, date("m", strtotime($nextbilldate)) + $monthsAhead, date("j", strtotime($nextbilldate)) + $daysAhead, date("Y", strtotime($nextbilldate)) + $yearsAhead));

                if (isset($newMonthTotals[$monthAndYear])) {
                    $newMonthTotals[$monthAndYear]['amount'] += $amount;
                    $newMonthTotals[$monthAndYear]['taxes']  += $taxes;
                    $newMonthTotals[$monthAndYear]['total']  += $amount + $taxes;
                    
                    if (isset($newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']])) {
                        $newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']][1] += 1;
                        $newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']][2] += $amount;
                        $newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']][3] += $taxes;
                        $newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']][4] += $amount + $taxes;
                    } else {
                        $newMonthTotals[$monthAndYear]['data'][$billingcycles[$paymentterm]['name']] = array(
                            $billingcycles[$paymentterm]['name'],
                            1,
                            $amount,
                            $taxes,
                            $amount + $taxes
                        );
                    }
                }

                switch ($billingcycles[$paymentterm]['time_unit']) {
                    case 'd':
                        $daysAhead += $billingcycles[$paymentterm]['amount_of_units'];
                        break;
                    case 'w':
                        $daysAhead += ($billingcycles[$paymentterm]['amount_of_units'] * 7);
                        break;
                    case 'm':
                        $monthsAhead += $billingcycles[$paymentterm]['amount_of_units'];
                        break;
                    case 'y':
                        $yearsAhead += $billingcycles[$paymentterm]['amount_of_units'];
                        break;
                }

                $currentMonthAndYear = date("Ym", mktime(0, 0, 0, date("m", strtotime($nextbilldate)) + $monthsAhead, date("j", strtotime($nextbilldate)) + $daysAhead, date("Y", strtotime($nextbilldate)) + $yearsAhead));
            } while ($currentMonthAndYear <= $maxMonthAndYear);
        }

        $this->SetDescription($this->user->lang('Displays revenue forecast'));

        $MonthsToDisplay =
             '<form id="report" method="GET">'
            .'    <div style="text-align:center">'
            .'        Months to display: '
            .'        <input type="text" name="amountOfMonths" id="amountOfMonths" size="2" value="'.CE_Lib::viewEscape($amountOfMonths).'" onkeydown="if (event.keyCode == 13) { event.preventDefault(); }"> '
            .$filter
            .'        <input type=button name=search class="btn" value=\''.$this->user->lang('Display').'\' onclick="ChangeTable(document.getElementById(\'amountOfMonths\').value, document.getElementById(\'currencycode\').value);">'
            .'    </div>'
            .'</form>'
            .'</br>'
            .'<script type="text/javascript">'
            .'    function ChangeTable(amountOfMonths, currencycode){'
            .'        location.href="index.php?fuse=reports&view=viewreport&controller=index&report=Revenue+Forecast&type=Revenue&amountOfMonths="+amountOfMonths+"&currencycode="+currencycode;'
            .'    }'
            .'    function LoadDetailsForMonthYear(month,data,currencycode){'
            .'        var url;';
        echo $MonthsToDisplay;

            $url = 'admin/index.php?noheaderfooter=1&getMonthDetails=1&month="+month+"&data="+data+"&currencycode="+currencycode+"&view='.CE_Lib::viewEscape($_REQUEST['view']).'&fuse='.CE_Lib::viewEscape($_REQUEST['fuse']).'&report='.urlencode($_REQUEST['report']).'&type='.CE_Lib::viewEscape($_REQUEST['type']);
            $url = mb_substr(CE_Lib::getSoftwareURL(), -1, 1) == "//" ? CE_Lib::getSoftwareURL().$url : CE_Lib::getSoftwareURL()."/".$url;

        $MonthsToDisplay =
            '        url ="'.$url.'";'
            .'        window.open(url, "", "top=100, left=100, width=800, height=600, scrollbars=yes");'
            .'    }'
            .'</script>';
        echo $MonthsToDisplay;

        $subGroup = array();

        foreach ($newMonthTotals as $monthAndYear => $monthData) {
            $encodedData = '';

            foreach ($monthData['data'] as $data) {
                if ($encodedData != '') {
                    $encodedData .= ';';
                }

                $encodedData .= implode('_', $data);
            }

            $subGroup[] = array(
                ($encodedData == '')? $monthAndYear : "<a href=\"javascript:LoadDetailsForMonthYear('".$monthAndYear."','".base64_encode($encodedData)."','".$currencyCode."');\">".$monthAndYear."</a>",
                $currency->format($currencyCode, $monthData['amount'], true, false),
                $currency->format($currencyCode, $monthData['taxes'], true, false),
                $currency->format($currencyCode, $monthData['total'], true, false)
            );
        }

        $this->reportData[] = array(
            "group"     => $subGroup,
            "groupname" => "",
            "label"     => array(
                $this->user->lang('Month and Year'),
                $this->user->lang('Projected Subtotal'),
                $this->user->lang('Projected Tax'),
                $this->user->lang('Projected Total Revenue')
            )
        );
    }

    function getMonthDetails($currencyCode)
    {
        if (isset($_GET['data'])) {
            $data = base64_decode($_GET['data']);
            $decodedData = explode(';', $data);
            $dataArray = array();

            foreach ($decodedData as $data) {
                $dataArray[] = explode('_', $data);
            }
        }

        include_once 'modules/billing/models/Currency.php';
        $currency = new Currency($this->user);

        echo "<H1>" . CE_Lib::viewEscape($_GET["month"]) . "</h1></br>";

        echo "<style>
            table {
                width:100%;
            }
            table, th, td {
                border: 1px solid black;
                border-collapse: collapse;
            }
            th, td {
                padding: 5px;
                text-align: left;
            }
            table#t01 tr:nth-child(even) {
                background-color: #eee;
            }
            table#t01 tr:nth-child(odd) {
               background-color:#fff;
            }
            table#t01 th	{
                background-color: white;
                color: black;
            }
            </style>";

        echo "<table id='t01'>";
        echo "<tr><th>". $this->user->lang('Type') . "</th><th>" . $this->user->lang('Items') . "</th><th>" . $this->user->lang('Subtotal') . "</th><th>" . $this->user->lang('Tax') . "</th><th>" . $this->user->lang('Total') . "</th></tr>";


        foreach ($dataArray as $data) {
            echo "<tr><td>" . $data[0] . "</td><td>" . $data[1] . "</td><td>" . $currency->format($currencyCode, $data[2], true, "NONE", true) . "</td><td>" . $currency->format($currencyCode, $data[3], true, "NONE", true). "</td><td>" . $currency->format($currencyCode, $data[4], true, "NONE", true). "</td></tr>";
        }

        echo "</table>";
    }
}
